package com.mantz_it.guitartunerlibrary;

import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.RectF;
import android.graphics.Typeface;

public class TunerSkinFilmLeader extends TunerSkin {

    private Paint linePaint;
    private Paint backgroundPaint;
    private Paint textPaint;
    private Paint needlePaint;
    private Paint inTunePaint;
    private Paint numberBoxPaint;
    private float centerX, centerY, radius;

    public TunerSkinFilmLeader() {
        init();
    }

    private void init() {
        backgroundPaint = new Paint();
        backgroundPaint.setColor(Color.BLACK);
        backgroundPaint.setStyle(Paint.Style.FILL);

        linePaint = new Paint();
        linePaint.setColor(Color.WHITE);
        linePaint.setStrokeWidth(4);
        linePaint.setStyle(Paint.Style.STROKE);
        linePaint.setAntiAlias(true);

        textPaint = new Paint();
        textPaint.setColor(Color.WHITE);
        textPaint.setTextAlign(Paint.Align.CENTER);
        textPaint.setAntiAlias(true);
        textPaint.setTypeface(Typeface.create(Typeface.MONOSPACE, Typeface.BOLD));

        needlePaint = new Paint();
        needlePaint.setColor(Color.RED);
        needlePaint.setStrokeWidth(12);
        needlePaint.setStyle(Paint.Style.STROKE);
        needlePaint.setAntiAlias(true);

        inTunePaint = new Paint();
        inTunePaint.setColor(Color.GREEN); // Or your preferred "in-tune" color
        inTunePaint.setStrokeWidth(16);
        inTunePaint.setStyle(Paint.Style.STROKE);
        inTunePaint.setAntiAlias(true);
        numberBoxPaint = new Paint();
        numberBoxPaint.setStyle(Paint.Style.FILL);
        numberBoxPaint.setAntiAlias(true);
    }

    @Override
    public void updateWidthAndHeight(int width, int height) {
        super.updateWidthAndHeight(width, height);
        this.centerX = width / 2f;
        this.centerY = height / 2f;
        this.radius = Math.min(width, height) * 0.4f; // Use 40% of the smaller dimension

        // Create a vertical linear gradient for the filmstrip background
        // It will go from GRAY at the top to DKGRAY at the bottom
        android.graphics.Shader backgroundGradient = new android.graphics.LinearGradient(
                centerX, 0, centerX, height,
                Color.GRAY, Color.DKGRAY,
                android.graphics.Shader.TileMode.CLAMP);

        // Apply the gradient to the paint
        numberBoxPaint.setShader(backgroundGradient);
    }

    // In TunerSkinFilmLeader.java
// REPLACE the entire draw method with this one.

// In TunerSkinFilmLeader.java
// REPLACE the entire draw method with this one.
// In TunerSkinFilmLeader.java
// REPLACE the entire draw method with this one.

    // In TunerSkinFilmLeader.java
// REPLACE the entire draw method with this one.

    @Override
    public void draw(Canvas canvas, GuitarTuner tuner) {
        // --- 1. Draw the Background and Filmstrip Box ---
        // First, draw the main black background
        canvas.drawRect(0, 0, width, height, backgroundPaint);

        // Now, draw the large vertical filmstrip box that goes behind everything.
        float filmstripWidth = width * 0.9f;
        float filmstripLeft = centerX - (filmstripWidth / 2);
        float filmstripRight = centerX + (filmstripWidth / 2);
        RectF filmstripRect = new RectF(filmstripLeft, 0, filmstripRight, height);
        canvas.drawRect(filmstripRect, numberBoxPaint); // Use the gray paint with gradient

        // Draw the "sprocket holes" on the sides of the screen
        float sprocketSize = width * 0.03f;
        float sprocketMargin = sprocketSize / 2;
        int numSprockets = (int) (height / (sprocketSize * 2.5f));

        for (int i = 0; i < numSprockets; i++) {
            float y = (i * sprocketSize * 2.5f) + sprocketSize;
            // Left side holes
            canvas.drawRect(sprocketMargin, y, sprocketMargin + sprocketSize, y + sprocketSize, numberBoxPaint);
            // Right side holes
            canvas.drawRect(width - sprocketMargin - sprocketSize, y, width - sprocketMargin, y + sprocketSize, numberBoxPaint);
        }


        // --- 2. Draw the static "scope" elements ---
        linePaint.setStrokeWidth(4);
        // Draw the TWO white circles
        canvas.drawCircle(centerX, centerY, radius, linePaint); // Outer circle
        canvas.drawCircle(centerX, centerY, radius * 0.9f, linePaint); // Inner circle

        // Draw the black crosshairs that extend across the filmstrip
        linePaint.setColor(Color.BLACK);
        canvas.drawLine(filmstripLeft, centerY, filmstripRight, centerY, linePaint);
        canvas.drawLine(centerX, 0, centerX, height, linePaint);
        // Reset paint color back to WHITE
        linePaint.setColor(Color.WHITE);


        // --- 3. Draw the subtle Countdown Numbers ---
        textPaint.setTextSize(radius * 0.25f);
        textPaint.setColor(Color.parseColor("#555555")); // Custom, very dark gray
        for (int i = 1; i <= 8; i++) {
            double angle = Math.PI / 4 * (i - 2);
            float numberRadius = radius * 0.7f; // Padded inward
            float x = centerX + (float) (Math.cos(angle) * numberRadius);
            float y = centerY + (float) (Math.sin(angle) * numberRadius) + textPaint.getTextSize() / 3;
            canvas.drawText(String.valueOf(9 - i), x, y, textPaint);
        }


        // --- 4. Draw the Note Name, Frequency, and Needle ---
        if (tuner.isValid()) {
            // --- 4a. Draw Note Name and Frequency ---
            // Decide the text color: GREEN if in tune, otherwise BLACK
            textPaint.setColor(tuner.isTuned() ? Color.GREEN : Color.BLACK);

            textPaint.setTextSize(radius * 0.5f); // Large text for the note
            canvas.drawText(tuner.pitchLetterFromIndex(tuner.getTargetPitchIndex()), centerX, centerY - radius * 0.2f, textPaint);

            textPaint.setTextSize(radius * 0.2f); // Smaller text for frequency
            String freqText = String.format(java.util.Locale.US, "%.2f Hz", tuner.getDetectedFrequency());
            canvas.drawText(freqText, centerX, centerY + radius * 0.4f, textPaint);

            // --- 4b. Draw the Rotating Black Needle ---
            float frequency = tuner.getDetectedFrequency();
            float targetFreq = tuner.getTargetFrequency();
            float cents = (float) (1200 * Math.log(frequency / targetFreq) / Math.log(2));
            cents = Math.max(-50.0f, Math.min(50.0f, cents));
            float rotationAngle = (cents / 50.0f) * 45.0f;

            canvas.save();
            canvas.rotate(rotationAngle, centerX, centerY);

            // Choose the needle color: GREEN if in tune, otherwise BLACK
            Paint needleColorPaint = tuner.isTuned() ? inTunePaint : backgroundPaint;
            needleColorPaint.setStrokeWidth(8);

            // Draw the needle as a vertical line pointing upwards from the center
            canvas.drawLine(centerX, centerY, centerX, centerY - radius * 0.85f, needleColorPaint);

            canvas.restore();

        } else {
            // If no signal, show dashes in BLACK to match the theme
            textPaint.setColor(Color.BLACK);
            textPaint.setTextSize(radius * 0.5f);
            canvas.drawText("-", centerX, centerY - radius * 0.2f, textPaint);
        }

        // --- 5. Draw the static "in-tune" marker at the top ---
        linePaint.setStrokeWidth(8);
        linePaint.setColor(Color.WHITE); // Make sure it's white
        canvas.drawLine(centerX, centerY - radius * 0.9f, centerX, centerY - radius * 1.1f, linePaint);

        // Finally, reset any lingering paint colors to be safe
        textPaint.setColor(Color.WHITE);
    }
}
